pro load_scene, inputfile, time=time, lambda=lambda, xystar=xystar, fstar=fstar, xyplanet=xyplanet, fplanet=fplanet, diskimage=diskimage, angdiam=angdiam, pixscale=pixscale

  ;This routine reads the output .fits file produced by exoVista
  ;and converts all quantities to flux at the same spectral resolution.
  ;Pixel coordinates of all objects are returned. Note that pixel coordinates
  ;are in 00LL format, where (0,0) is the lower-left corner of the lower-left
  ;pixel (and (0.5,0.5) is the center of the lower-left pixel). The star
  ;should be located at exactly (npix/2,npix/2)--the intersection of
  ;the central 4 pixels.

  ;--- INPUTS ---
  ;inputfile = filename and path of fits file containing scene
  ;time = desired time (default = 0)

  ;--- OUTPUTS --- 
  ;lambda: wavelength vector (microns)
  ;xystar: location of star (pixels)
  ;fstar: stellar flux vector (Jy)
  ;xyplanet: Nplanets x 2 array of planet locations (pixels)
  ;fplanet: Nplanets x nlambda array of planet fluxes (Jy)
  ;diskimage: npix x npix x nlambda disk image cube (Jy per pix)
  ;angdiam: angular diameter of star (mas)
  ;pixscale: pixel scale (mas)
  
  if n_elements(time) eq 0 then time=0. ;default

  ;Define extension numbers
  lam_ext = 0
  disklam_ext = 1
  disk_ext = 2
  star_ext = 3
  planet_ext = 4 ;first planet extension
  h = headfits(inputfile,exten=0) ;read header of first extension
  n_ext = sxpar(h,'N_EXT') ;get the largest extension #

  ;Get wavelength array
  lambda = readfits(inputfile,h,ext=lam_ext) ;read wavelength extension
  nlambda = n_elements(lambda)
  
  ;STEP 1: STAR
  ;Need to determine x, y, and fstar
  xystar = fltarr(2)
  fstar = fltarr(nlambda)
  d = readfits(inputfile,h,exten_no=star_ext)  ;read star extension
  angdiam = sxpar(h,'ANGDIAM') ;user might want to know this
  pixscale = sxpar(h,'PXSCLMAS')
  t = reform(d[0,*]) ;time vector
  x = reform(d[1,*]) ;heliocentric x location vector (pix)
  y = reform(d[2,*]) ;heliocentric y location vector (pix)
  xystar[0] = x[0] ;pick the first entry by default
  xystar[1] = y[0]

  ;If the fits file contains a vector of times, interpolate...
  if n_elements(t) gt 1 then xystar[0] = interpol(x,t,time,/quadratic) ;use quadratic interpolation for positions
  if n_elements(t) gt 1 then xystar[1] = interpol(y,t,time,/quadratic) ;use quadratic interpolation for positions

  ;Now stellar flux...
  fstar[*] = reform(d[15:15+nlambda-1,0]) ;grab the stellar flux of first time entry
  if n_elements(t) gt 1 then begin ;if fits file contains a vector of times, interpolate...
     for i=0,nlambda-1 do fstar[i] = interpol(reform(d[15+i,*]),t,time,/quadratic) ;interpolate time for each lambda
  endif
  
  ;STEP 2: PLANETS
  ;Need to determine x, y, and fplanet
  nplanets = n_ext - 3
  xyplanet = fltarr(nplanets,2)
  fplanet = fltarr(nplanets,nlambda)
  for ip = 0, nplanets-1 do begin ;loop over all planets
     d = readfits(inputfile,h,exten_no=planet_ext+ip) ;read the appropriate extension
     t = reform(d[0,*]) ;time vector
     x = reform(d[1,*]) ;heliocentric x position vector (pix)
     y = reform(d[2,*]) ;heliocentric y position vector (pix)
     xyplanet[ip,0] = x[0] ;pick the first entry by default 
     xyplanet[ip,1] = y[0]

     ;If the fits file contains a vector of times, interpolate...
     if n_elements(t) gt 1 then xyplanet[ip,0] = interpol(x,t,time,/quadratic) ;use quadratic interpolation for positions
     if n_elements(t) gt 1 then xyplanet[ip,1] = interpol(y,t,time,/quadratic) ;use quadratic interpolation for positions

     ;Now the planet flux...
     contrast = reform(d[15:15+nlambda-1,0]) ;data array contains planet *contrast*
     fplanet[ip,*] = contrast * fstar ;convert to flux, use first time entry by default
     if n_elements(t) gt 1 then begin ;if fits file contains a vector of times, interpolate...
        for i=0,nlambda-1 do begin
           contrast = interpol(reform(d[15+i,*]),t,time,/quadratic) ;interpolate planet *contrast* in time
           fplanet[ip,i] = contrast * fstar[i]                      ;convert to flux
        endfor
     endif
  endfor
  
  ;STEP 3: DISK
  lambda_disk = readfits(inputfile,h,exten_no=disklam_ext) ;disk wavelengths
  temp = readfits(inputfile,h,exten_no=disk_ext) ;contrast cube + precision
  nlambda_disk = n_elements(lambda_disk)
  contrast = temp[*,*,0:(nlambda_disk-1)]
  cprecision = temp[*,*,nlambda_disk]
  ;Interpolate the disk image cube to the desired wavelength spacing
  ;method 1: this is slow, so I don't use it.
  ;disk_image = fltarr(npix,npix,nlambda)
  ;for ix=0,npix-1 do begin
  ;   for iy=0,npix-1 do begin
  ;      disk_image[ix,iy,*] = interpol(contrast[ix,iy,*],lambda_disk,lambda,/quadratic)
  ;   endfor
  ;endfor
  ;method 2: this is fast and I think more accurate
  ;interpolate(d,indices) will interpolate the right-most
  ;index of d when provided only 1 set of indices, which is
  ;what we want.
  lambda_indices = reform(value_locate(lambda_disk,lambda)) ;the indices in lambda of each value of disk_lambda
  lambda_indices += (alog(lambda)-alog(lambda_disk[lambda_indices]))/(alog(lambda_disk[lambda_indices+1])-alog(lambda_disk[lambda_indices])) ;index in log lambda space (have to add on fractional indices)
  diskimage = interpolate(contrast,lambda_indices,cubic=-0.5) ;we can smoothly interpolate contrast
  for i=0,nlambda-1 do diskimage[*,*,i] *= fstar[i] ;now that contrast is interpolated, multiply by fstar to get disk flux cube


  
end
