pro load_occurrence_rates, occrate, occrate_lnrmin, occrate_lnamin, occrate_dlnr, occrate_dlna, npix, a_array=a_array, r_array=r_array, ll=ll, ul=ul, sag13=sag13,mass=mass,quiet=quiet

;load_occurrence_rates.pro
;This procedure generates a 2D array of occurrence rates in
;(R_planet,a_planet) space. The bin size of each pixel is controlled
;by the keyword npix and has constant dln(R) dln(P). Occurrence rates
;returned are integrated over the bin.
    
;By default, the nominal occurrence rates of Dulz & Plavchan
;are returned.  Users can set the ll or ul keywords to return
;the "1-sigma" lower and upper limits, respectively.
;Users can also set the sag13 keyword to return the pure SAG13
;analytically-calculated occurrence rates.


  
;OUTPUTS
;occrates: a 2D vector (x axis = semi-major axis, y axis = planet radius)
;occrate_lnrmin: the minimum of ln(r) used for the grid
;occrate_lnamin: the minimum of ln(a) used for the grid
;occrate_dlnr: the value of dlnr (pixel scale along y axis)
;occrate_dlna: the value of dlna (pixel scale along x axis)
;npix: the # of pixels of the output occrate array along 1 axis
  
;OPTIONAL OUTPUTS
;a_array: a 2D array of semi-major axis value of every pixel
;r_array: a 2D array of planet radius values of every pixel
  
;OPTIONAL INPUTS
;ll: setting this keyword returns the pessimistic occurrence rates
;ul: setting this keyword returns the optimistic occurrence rates
;sag13: setting this keyword returns the pure SAG13 analytic fit occurrence rates   
;MASS: If the mass keyword is set, it returns the Dulz & Plavchan occurrence
;rates as a histogram in (M_planet,a_planet). The input occrate_lnrmin
;is assumed to express the minimum of ln(Mass) used for the grid,
;the input occrate_dlnr is assumed to express the value of dlnMass to
;be used for the grid, and the r_array output contains the masses in
;Earth masses.


;Notes:
;- the Dulz & Plavchan occurrence rate files are used to set the grid
;  spacing, even if SAG13 are requested.
;- Dulz & Plavchan occ rates are not analytic, so we must interpolate
;  their results to the desired resolution


  
;-----------------------
;CHECK SYNTAX
if keyword_set(ul) and keyword_set(ll) then stop,'Cannot request both lower limit and upper limit on occurrence rates.'
if keyword_set(mass) and keyword_set(SAG13) then stop,'Cannot request SAG13 occurrence rates in (mass, semi-major axis) space.'
;-----------------------


;-----------------------
;MAIN BODY

if n_elements(npix) eq 0 then npix = long(500) ;resolution for occurrence rate map

;Read in the files and perform some conversions
tag = 'Radius'
if keyword_set(mass) then begin
   tag = 'Mass'
   if not keyword_set(quiet) then print,'  Retrieving occurrence rates in non-standard space of (mass, semi-major axis)...'
endif
filename = 'occurrence_rates/NominalOcc_'+tag+'.csv'
if keyword_set(ll) then filename = 'occurrence_rates/PessimisticOcc_'+tag+'.csv'
if keyword_set(ul) then filename = 'occurrence_rates/OptimisticOcc_'+tag+'.csv'
if not keyword_set(sag13) and not keyword_set(quiet) then print,'  Loading occurrence rates: '+filename

d = read_csv(filename)
tempoccrate = d.field1
rmin = d.field5                 ;in Earth radii
r = d.field6           ;in Earth radii
rmax = d.field7        ;in Earth radii


pmin = d.field2/365.25          ;convert from days to years
p = d.field3/365.25    ;convert from days to years
pmax = d.field4/365.25 ;convert from days to years
amin = pmin^(2./3.)    ;in AU
a = p^(2./3.)          ;in AU
amax = pmax^(2./3.)    ;in AU
if keyword_set(mass) then begin ;for some reason the mass input files contain semi-major axis in AU, not period
   amin = d.field2
   a = d.field3
   amax = d.field4
   pmin = amin^1.5
   p = a^1.5
   pmax = amax^1.5
endif


;Create x and y-axis vectors
pminvec = pmin[uniq(pmin,sort(pmin))]
pvec = p[uniq(p,sort(p))]
pmaxvec = pmax[uniq(pmax,sort(pmax))]

aminvec = amin[uniq(amin,sort(amin))]
avec = a[uniq(a,sort(a))]
amaxvec = amax[uniq(amax,sort(amax))]

rminvec = rmin[uniq(rmin,sort(rmin))]
rvec = r[uniq(r,sort(r))]
rmaxvec = rmax[uniq(rmax,sort(rmax))]


;Turn the occ rate list into a 2D array
;using the x and y-axis vectors
occrate = fltarr(n_elements(avec),n_elements(rvec))
for i=0,n_elements(tempoccrate)-1 do begin
   ia = where(avec eq a[i])
   ir = where(rvec eq r[i])
   occrate[ia[0],ir[0]] = tempoccrate[i]
endfor


;Convert occurrence rates to d^2 occrate / dlnp dlnr
lnr = alog(rvec)
dlnr = lnr[1:n_elements(lnr)-1] - lnr[0:n_elements(lnr)-2]
if abs(max(dlnr)-min(dlnr))/abs(min(dlnr)) gt 1e-2 then stop,'Radius spacing must be constant in ln space'
dlnr = alog(rvec[1])-alog(rvec[0])
lnp = alog(pvec)
dlnp = lnp[1:n_elements(lnp)-1] - lnp[0:n_elements(lnp)-2]
if abs(max(dlnp)-min(dlnp))/abs(min(dlnp)) gt 1e-2 then stop,'Period spacing must be constant in ln space'
dlnp = alog(pvec[1])-alog(pvec[0])
occrate /= (dlnr * dlnp)

;Now interpolate to a much higher resolution
occrate = congrid(occrate,npix,npix,cubic=-0.5) > 0

;Calculate new x and y-axis vectors
dlnp_out = (alog(max(pmaxvec)) - alog(min(pminvec)))/npix
lnp_out = findgen(npix)*dlnp_out + alog(min(pminvec))
dlnr_out = (alog(max(rmaxvec)) - alog(min(rminvec)))/npix
lnr_out = findgen(npix)*dlnr_out + alog(min(rminvec))
dlna_out = (alog(max(amaxvec)) - alog(min(aminvec)))/npix
lna_out = findgen(npix)*dlna_out + alog(min(aminvec))

;Change occ rates back to being integrated over a bin
occrate *= (dlnr_out * dlnp_out)

;Load SAG13 if desired
if keyword_set(sag13) then begin
   if not keyword_set(quiet) then print,'  Loading occurrence rates: SAG13 analytic fit'
   occrate = sag13_eta_grid(rmin=min(rmin),rmax=max(rmax),amin=min(amin),amax=max(amax),nbins=npix+1)
   if not keyword_set(ll) and not keyword_set(ul) then occrate = occrate[*,*,1] ;nominal
   if keyword_set(ll) then occrate = occrate[*,*,0] ;lower limit
   if keyword_set(ul) then occrate = occrate[*,*,2] ;upper limit
endif

if max(occrate) gt 1 then stop,'Max occurrence rate is > 1, so Monte Carlo routine will not work.  Increase value of npix.'

;Calculate a few remaining outputs
a = exp(lna_out)
r = exp(lnr_out)
a_array = congrid([[a],[a]],npix,npix)
r_array = rotate(congrid([[r],[r]],npix,npix),1)

occrate_lnrmin = min(lnr_out)
occrate_lnamin = min(lna_out)
occrate_dlnr = dlnr_out
occrate_dlna = dlna_out


end 

















