pro getkurucz, Teff, logg, metallicity, lambda, Bnu, logg_model=logg_model, teff_model=teff_model, Bnucont=Bnucont, quiet=quiet, path=path

  ;For a given Teff, log(g), and metallicity, retrieves the
  ;Kurucz & Castelli ATLAST9 stellar atmosphere models.
  ;Ideally it would interpolate these, but right now it just
  ;picks the closest match.
  ;As recommended by the STScI web site discussing synphot,
  ;we use the models with vturb=2 km/s and Delta(log(tau_ross))=0.125
  ;Currently we only read in the zero metallicity file, though more files
  ;could be added to select from in the future...

  
  ;INPUTS
  ;Teff: stellar effective temp (K)
  ;logg: log of surface gravity
  ;metallicity: currently only zero is allowed
  
  ;OUTPUTS
  ;lambda: wavelength vector (nm)
  ;Bnu: surface brightness (erg s^-1 cm^-2 Hz^-1 steradian^-1)

  ;OPTIONAL OUTPUTS
  ;Bnucont: continuum surface brightness w/out line abosrptions (erg s^-1 cm^-2 Hz^-1 steradian^-1)
  ;logg_model: best matching value of logg in the grid
  ;teff_model: best matching value of teff in the grid
  
  
  ;Vectors of logg and teff in the Kurucz model grid...
  gvec = [0.0,0.5,1.0,1.5,2.0,2.5,3.0,3.5,4.0,4.5,5.0]
  tvec = [3500,3750,4000,4250,4500,4750,5000,5250,5500,5750,6000,6250,6500,6750,7000,7250,7500,7750,8000,8250,8500,8750,9000,9250,9500,9750,10000,10250,10500,10750,11000,11250,11500,11750,12000,12250,12500,12750,13000,14000,15000,16000,17000,18000,19000,20000,21000,22000,23000,24000,25000,26000,27000,28000,29000,30000,31000,32000,33000,34000,35000,36000,37000,38000,39000,40000,41000,42000,43000,44000,45000,46000,47000,48000,49000,50000]
  

  ;First, some error handling...
  if teff gt max(tvec) then print,'Teff too large.'
  if teff lt min(tvec) then print,'Teff too small.'
  if logg gt max(gvec) then print,'logg too large.'
  if logg lt min(gvec) then print,'logg too small.'
  if metallicity ne 0 then stop,'Only [+0.0] metallicity currently allowed.'

  ;Open model grid file
  file = 'fp00k2odfnew.pck'
  if n_elements(path) ne 0 then file = path+file

  openr, lun, file, /get_lun
  ;Skip past the header
  nheaderlines = 22
  SKIP_LUN, lun, nheaderlines, /LINES  

  
  ;Read the wavelengths (in nm)
  line = ''
  templine = ''
  while not strmatch(templine,'*TEFF*',/FOLD_CASE) do begin
     line = line + ' ' + templine
     readf, lun, templine
  endwhile
  lambda = float(strsplit(line,' ',/extract)) ;nm
  nl = n_elements(lambda)

    
  ;Must keep logg_model from going out of bounds
  teff_model = teff
  logg_model = logg
  if teff_model ge 6250 then logg_model = logg_model > 0.5
  if teff_model ge 7750 then logg_model = logg_model > 1.0
  if teff_model ge 8500 then logg_model = logg_model > 1.5
  if teff_model ge 9250 then logg_model = logg_model > 2.0
  if teff_model ge 12000 then logg_model = logg_model > 2.5
  if teff_model ge 20000 then logg_model = logg_model > 3.0
  if teff_model ge 27000 then logg_model = logg_model > 3.5
  if teff_model ge 32000 then logg_model = logg_model > 4.0
  if teff_model ge 40000 then logg_model = logg_model > 4.5
  if teff_model ge 50000 then logg_model = logg_model > 5.0


  ;logg_model = logg_model > (min(gvec)*1.01) ;don't let it go less than the min value
  ;logg_model = logg_model < (max(gvec)*0.99) ;don't let it excede the maximum value
  ;teff_model = teff_model > (min(tvec)*1.01)
  ;teff_model = teff_model < (max(tvec)*0.99)
  
  ;Get the fractional indices of the desired logg and teff
  gindex = value_locate(gvec,logg_model)
  g2 = gindex+1 < (n_elements(gvec)-1)
  g1 = g2-1
  gindex += (logg_model - gvec[gindex])/(gvec[g2]-gvec[g1])
  tindex = value_locate(tvec,teff_model)
  t2 = tindex+1 < (n_elements(tvec)-1)
  t1 = t2-1
  tindex += (teff_model - tvec[tindex])/(tvec[tindex+1]-tvec[tindex])



  Bnu_array = fltarr(nl,n_elements(gvec),n_elements(tvec))

  
  while not eof(lun) do begin
     while not strmatch(templine,'*TEFF*',/FOLD_CASE) do readf,lun,templine
     templine = strsplit(templine,' ',/extract)

     tempt = float(templine[1])
     tempg = float(templine[3])
     jt = where(tempt eq tvec)
     jg = where(tempg eq gvec)
     if jt[0] eq -1 or jg[0] eq -1 then stop,'Error.'

     ;Read out the data
     ;Get Bnu
     line = ''
     templine = ''
     while strlen(line)/10 lt nl do begin
        readf, lun, templine
        line = line + templine
     endwhile
     Bnu = double(strmid(line,lindgen(nl)*10,10))
     
     ;Get Fnucont
     line = ''
     templine = ''
     while strlen(line)/10 lt nl do begin
        readf, lun, templine
        line = line + templine
     endwhile
     Bnucont = double(strmid(line,lindgen(nl)*10,10))

     Bnu *= 4.                  ;convert from Hnu to Bnu (see note below)
     Bnucont *= 4.

  ;//IMPORTANT NOTE: The Kurucz flux data is in units of erg s^-1 cm^-2 Hz^-1 steradian^-1.  This
  ;//is the same units as surface brightness, so we might think that Kurucz is providing Bnu, where
  ;//Fnu = pi * Bnu * (Rstar / d)^2 is the flux Fnu for a star of radius Rstar at distance d.  But in
  ;//fact, for whatever reason, Kurucz is providing the quantity (pi * Bnu) / (4 * pi).  This is the
  ;//surface flux per unit solid angle--this must be important in stellar atmospheres or something.
  ;//Anyway, to calculate Bnu, which is what this routine returns, we multiply Kurucz's numbers by 4.
  ;//If this sounds crazy to you, I note that this agrees with the SYNPHOT Data User's Guide, which
  ;//can be found at http://www.stsci.edu/hst/HST_overview/documents/synphot/AppA_Catalogs9.html

     
     Bnu_array[*,jg,jt] = Bnu ;save into the master array
     
  endwhile



  ;Now interpolate to the desired logg and teff
  Bnu = interpolate(Bnu_array, gindex, tindex,cubic=-0.5)

  free_lun,lun
  
end
