function get_stellar_flux, s, lambda, path=path

  ;INPUTS
  ;s: star structure
  ;lambda: desired wavelengths (microns)

  ;OUTPUTS
  ;fstar: stellar flux in Jy
  
  Tstar = s.teff
  logg = s.logg
  metallicity = 0.0
  Rstar = s.Rstar
  dstar = s.dist
  RstarAU = Rstar * 0.00465047  ;radius of star in AU
  dstarAU = s.dist * 206265.    ;distance to star in AU

  
  ;Load the appropriate Kurucz stellar atmosphere model
  ;klambda in units of nm
  ;kBnu is in units of erg s^-1 cm^-2 Hz^-1 steradian^-1
  getkurucz, Tstar, logg, metallicity, klambda, kBnu, path=path    
  klambda *= 0.001 ;convert to microns

  ;Interpolate to high spectral resolution
  specres = 10000 ;internal spectral resolution

  dlnlambda = 1./specres
  lnlambdamax = alog(max(klambda))
  lnlambdamin = alog(min(klambda))
  nlambda = ceil((lnlambdamax-lnlambdamin)/dlnlambda)
  interplambda = (findgen(nlambda))/(nlambda-1)*(lnlambdamax-lnlambdamin)+lnlambdamin
  interplambda = exp(interplambda)

  ;Now add in the "transition wavelengths" so we can
  ;accurately bin things. "Transition wavelengths" are
  ;the wavelengths that divide the spectral channels.
  transition_lambda = sqrt(lambda[0:n_elements(lambda)-2]*lambda[1:n_elements(lambda)-1])
  transition_lambda = [(transition_lambda[0]/lambda[1])*lambda[0],transition_lambda,(transition_lambda[n_elements(transition_lambda)-1]/lambda[n_elements(lambda)-2])*lambda[n_elements(lambda)-1]]
  interplambda = [lambda,transition_lambda] ;add these to the array
  k = sort(interplambda) ;sort them
  interplambda = interplambda[k]
  k = uniq(interplambda) ;make sure wavelengths are unique
  interplambda = interplambda[k]  
  interpBnu = interpol(kBnu, klambda, interplambda, /spline)

  ;Now integrate over each spectral channel
  tempnu = (2.998e14)/interplambda                                                              ;frequency in Hz
  tempdnu = abs(tempnu[1:n_elements(tempnu)-1]-tempnu[0:n_elements(tempnu)-2])                ;will use this for integration
  integratedBnu = fltarr(n_elements(lambda))
  for ilambda=0,n_elements(lambda)-1 do begin
     j = where(interplambda ge transition_lambda[ilambda] and interplambda lt transition_lambda[ilambda+1])
     integratedBnu[ilambda] = total(interpBnu[j] * tempdnu[j]) / total(tempdnu[j]) ;integrate over spectral channel and divide by total dnu
  endfor  
  
  ;Convert to stellar flux in units of erg s^-1 cm^-2 Hz^-1 
  fstar = !dpi * integratedBnu * (RstarAU / dstarAU)^2
  ;Convert to Jy
  fstar *= 1e23


  ;The above normalizes the star based on the Kurucz model
  ;and B-V derived stellar radii. This may not match reality
  ;very well. So instead, we now normalize based on observed V band.
  TYCHO_V_ZP_JY = 3653.89 ;Jy,  the zero point is for the TYCHO filter below
  observedVJy = TYCHO_V_ZP_JY * 10.^(-0.4 * s.Vmag) ;Jy

  filterfile = path+'TYCHO_TYCHO.V.dat'
  f = read_ascii(filterfile,comment_symbol='#')
  f = f.field1
  l = reform(f[0,*])*0.0001 ;filter wavelength in microns
  f = reform(f[1,*])                    ;filter throughput
  interpolated_model = interpol(fstar,lambda,l,/nan)
  modelVJy = total(interpolated_model * f) / total(f)
    
  fstar *= (observedVJy/modelVJy)
  
  return, fstar
  
end
