pro generate_solarsystem, s, distance=distance, inclination=inclination, pa=pa, nzodis=nzodis

  ;s is the return data structure that can be sent to generate_scene

  if n_elements(distance) eq 0 then distance = 10.       ;default of 10 pc
  if n_elements(inclination) eq 0 then inclination = 60. ;default of 60 deg from face-on
  if n_elements(pa) eq 0 then PA = 0.
  if n_elements(nzodis) eq 0 then nzodis = 1. ;default of 1 zodi for all disks
  
  ;Constants
  grav_const = 4*!dpi*!dpi               ;AU^3 yr^-2 MSun^-1
  c = 2.998e8 / 1.496e11 * 365.25 * 24. * 60. * 60. ;AU yr^-1  

  
  ;Set up the star structure
  s = {starID:long(0), HIP:long(0), TYC:'', dist:0., M_V:0., vmag:0., bmag:0., umag:0., rmag:0., imag:0., jmag:0., hmag:0., kmag:0., type:'', lstar:0., logg:0., teff:0., angdiam:0., mass:0., rstar:0.}

  ;Fill in the Sun's values
  s.starID = -1
  s.HIP = -1
  s.TYC = 'SUN'
  s.dist = distance ;distance in pc
  s.M_V = 4.83 ;absolute V band magnitude
  s.vmag = s.M_V-5 + 5*alog10(s.dist) ;apparent V band mag
  s.bmag = s.vmag + 0.65 ;apparent B band mag
  s.umag = !values.f_nan ;etc.
  s.rmag = !values.f_nan
  s.imag = !values.f_nan
  s.jmag = !values.f_nan
  s.hmag = !values.f_nan
  s.kmag = !values.f_nan
  s.type = 'G2V'
  s.lstar = 1.0 ;solar luminosities
  s.logg = 4.4
  s.teff = 5778. ;K
  s.angdiam = 0.465/(s.dist/10.) ;mas
  s.mass = 1.0 ;solar masses
  s.rstar = 1.0 ;solar radii


  ;Add the planet structure
  maxnplanets = 30
  planet_entry = {M:0.0, R:0.0, albedo_file:'', a:0.0, e:0.0, i:0.0, longnode:0.0, argperi:0.0, meananom:0.0}
  planet_entries = replicate(planet_entry,maxnplanets)
  temp = replicate(create_struct('planet',planet_entries,'PA',0.,'I',0.,s[0]),1) ;make new array of structures w/ planet tag
  struct_assign,s,temp                                                                ;place values from stars array structure into temp
  s = temp                                                                            ;overwrite
  s.pa = pa
  s.i = inclination

  ;Now populate the planet params
  i = 0

  ;We get J2000 orbital parameters from nssdc.gsfc.nasa.gov fact sheets

  dir = './geometric_albedo_files/'
  ;Venus
  s.planet[i].M = 0.815
  s.planet[i].R = 0.949
  s.planet[i].albedo_file = dir+'Venus.txt'
  s.planet[i].a = 0.72333199
  s.planet[i].e = 0.00677323
  s.planet[i].i = 3.39471
  s.planet[i].longnode = 76.68069
  ;argperi = longitude of pericenter - longitude of ascending node
  s.planet[i].argperi = 131.53298 - s.planet[i].longnode           ;w
  ;mean anomaly = mean longitude - longitude of pericenter
  s.planet[i].meananom = 181.97973 - s.planet[i].longnode - s.planet[i].argperi

  ;Earth
  i++
  s.planet[i].M = 1.0
  s.planet[i].R = 1.0
  s.planet[i].albedo_file = dir+'Earth.txt'
  s.planet[i].a = 1.00000011  
  s.planet[i].e = 0.01671022  
  s.planet[i].i = 0.00005
  s.planet[i].longnode = -11.26064 ;omega
  ;argperi = longitude of pericenter - longitude of ascending node
  s.planet[i].argperi = 102.94719 - s.planet[i].longnode           ;w
  ;mean anomaly = mean longitude - longitude of pericenter
  s.planet[i].meananom = 100.46435 - s.planet[i].longnode - s.planet[i].argperi
  
  ;Mars
  i++
  s.planet[i].M = 0.107
  s.planet[i].R = 0.532
  s.planet[i].albedo_file = dir+'Mars.txt'
  s.planet[i].a = 1.52366231  
  s.planet[i].e = 0.09341233  
  s.planet[i].i = 1.85061
  s.planet[i].longnode = 49.57854 ;omega
  ;argperi = longitude of pericenter - longitude of ascending node
  s.planet[i].argperi = 336.04084 - s.planet[i].longnode           ;w
  ;mean anomaly = mean longitude - longitude of pericenter
  s.planet[i].meananom = 355.45332 - s.planet[i].longnode - s.planet[i].argperi

  ;Jupiter
  i++
  s.planet[i].M = 317.83
  s.planet[i].R = 11.209
  s.planet[i].albedo_file = dir+'Jupiter.txt'
  s.planet[i].a = 5.20336301  
  s.planet[i].e = 0.04839266  
  s.planet[i].i = 1.30530
  s.planet[i].longnode = 100.55615 ;omega
  ;argperi = longitude of pericenter - longitude of ascending node
  s.planet[i].argperi = 14.75385 - s.planet[i].longnode           ;w
  ;mean anomaly = mean longitude - longitude of pericenter
  s.planet[i].meananom = 34.40438 - s.planet[i].longnode - s.planet[i].argperi

  ;Saturn
  i++
  s.planet[i].M = 95.16
  s.planet[i].R = 9.449
  s.planet[i].albedo_file = dir+'Saturn.txt'
  s.planet[i].a = 9.53707032  
  s.planet[i].e = 0.05415060  
  s.planet[i].i = 2.48446
  s.planet[i].longnode = 113.71504 ;omega
  ;argperi = longitude of pericenter - longitude of ascending node
  s.planet[i].argperi = 92.43194 - s.planet[i].longnode           ;w
  ;mean anomaly = mean longitude - longitude of pericenter
  s.planet[i].meananom = 49.94432 - s.planet[i].longnode - s.planet[i].argperi

  ;Uranus
  i++
  s.planet[i].M = 14.54
  s.planet[i].R = 4.007
  s.planet[i].albedo_file = dir+'Uranus.txt'
  s.planet[i].a = 19.19126393  
  s.planet[i].e = 0.04716771  
  s.planet[i].i = 0.76986
  s.planet[i].longnode = 74.22988 ;omega
  ;argperi = longitude of pericenter - longitude of ascending node
  s.planet[i].argperi = 170.96424 - s.planet[i].longnode           ;w
  ;mean anomaly = mean longitude - longitude of pericenter
  s.planet[i].meananom = 313.23218 - s.planet[i].longnode - s.planet[i].argperi

  ;Neptune
  i++
  s.planet[i].M = 17.15
  s.planet[i].R = 3.883
  s.planet[i].albedo_file = dir+'Neptune.txt'
  s.planet[i].a = 30.06896348  
  s.planet[i].e = 0.00858587  
  s.planet[i].i = 1.76917
  s.planet[i].longnode = 131.72169 ;omega
  ;argperi = longitude of pericenter - longitude of ascending node
  s.planet[i].argperi = 44.97135 - s.planet[i].longnode           ;w
  ;mean anomaly = mean longitude - longitude of pericenter
  s.planet[i].meananom = 304.88003 - s.planet[i].longnode - s.planet[i].argperi




  ;We call generate_disks just to get the disk structure added
  generate_disks,s,/quiet,ncomponents=3
  ;reset all disk params to zero
  tags = tag_names(s.disk[0])
  for i=0,n_elements(tags)-1 do s.disk[*].(i) = 0
  
  ;Zodiacal component
  i = 0
  s.disk[i].nzodis = nzodis
  s.disk[i].r = 3.
  s.disk[i].dror = 0.1
  s.disk[i].rinner = 0.01
  beta = 0.5
  alpha = beta * grav_const * s.MASS / c          ;quantity related to PR drag to be used later
  tpr = s.disk[i].r^2 / (4 * alpha)               ;yr
  torbit = sqrt(s.disk[i].r^3 / s.MASS)           ;yr
  tcoll = torbit / (4*!dpi*s.disk[i].nzodis*1e-7) ;yr
  s.disk[i].eta = tpr / tcoll
  s.disk[i].hor = 0.1

  ;Cold Classical KB component
  ;params based on Kuchner & Stark 2010
  i++
  s.disk[i].nzodis = nzodis * (0.163 / (0.163+0.797))
  s.disk[i].r = 43.75
  s.disk[i].dror = 0.028
  s.disk[i].rinner = (1.1 * (5.2 * (1+0.04))) ;based on Jupiter
  s.disk[i].hor = tan(1.5*!dpi/180.)

  ;Cold Classical KB component
  ;params based on Kuchner & Stark 2010
  i++
  s.disk[i].nzodis = nzodis * (0.797 / (0.163+0.797))
  s.disk[i].r = 42.5
  s.disk[i].dror = 0.176
  s.disk[i].rinner = (1.1 * (5.2 * (1+0.04))) ;based on Jupiter
  s.disk[i].hor = tan(13.0*!dpi/180.)

  ;now go back and update eta values for both components
  i = 1
  beta = 0.5
  alpha = beta * grav_const * s.MASS / c          ;quantity related to PR drag to be used later
  tpr = s.disk[i].r^2 / (4 * alpha)               ;yr
  torbit = sqrt(s.disk[i].r^3 / s.MASS)           ;yr
  tcoll = torbit / (4*!dpi*s.disk[i].nzodis*1e-7) ;yr
  s.disk[i].eta = tpr / tcoll
  
  i = 2
  beta = 0.5
  alpha = beta * grav_const * s.MASS / c          ;quantity related to PR drag to be used later
  tpr = s.disk[i].r^2 / (4 * alpha)               ;yr
  torbit = sqrt(s.disk[i].r^3 / s.MASS)           ;yr
  tcoll = torbit / (4*!dpi*(s.disk[1].nzodis + s.disk[2].nzodis)*1e-7) ;yr
  s.disk[i].eta = tpr / tcoll
  
  ;adopt hedman & stark 2015 Saturn G ring fit for all disk components
  s.disk[*].g = [0.995, 0.665, 0.035]
  s.disk[*].w = [0.643, 0.176, 0.181]

  
end
