pro generate_scene, stars, pixscale=pixscale, npix=npix, lambdamin=lambdamin, lambdamax=lambdamax, specres = specres, specrdisk=specrdisk, timemax=timemax, dt=dt, seed=seed, output_dir=output_dir, quiet=quiet, lambda_disk=lambda_disk, iwa=iwa, iwa_tol=iwa_tol

  
  if n_elements(pixscale) eq 0 then pixscale = 0.002        ;arcsec
  if n_elements(iwa) eq 0 then iwa = 0.015 ;pixels inside of iwa are only solved to iwa_tol tolerance  
  if n_elements(iwa_tol) eq 0 then iwa_tol = 0.1 ;pixels inside of iwa are only solved to iwa_tol tolerance
  ;The above **greatly** speeds up the calculations. 30 mas is 
  ;LUVOIR-A's IWA at 300 nm, so 20 mas is pretty small and
  ;this should not significantly impact how the outputs are used.
  pixscale_arcsec = pixscale
  if n_elements(npix) eq 0 then npix = 250
  if n_elements(output_dir) eq 0 then print,'No output directory provided. Using ./output/'  
  if n_elements(output_dir) eq 0 then output_dir = 'output'
  if strmid(output_dir,0,1,/reverse_offset) ne '/' then output_dir = output_dir + '/'
  file_mkdir,output_dir
  if n_elements(seed) eq 0 then seed = systime(/sec) ;by default the results are not reproducible
  input_seed = seed ;save this seed
  nthreads = !CPU.HW_NCPU        ; Number of CPU threads
  if n_elements(flatSB) eq 0 then flatSB = 0
  
  routinename='generate_scene'
  exovistapath=routine_info(routinename,/source)
  exovistapath=strmid(exovistapath.path,0,strlen(exovistapath.path)-strlen(routinename)-4)

  routinename='writefits'
  resolve_routine, routinename
  astronpath=routine_info(routinename,/source)
  astronpath=strmid(astronpath.path,0,strlen(astronpath.path)-strlen(routinename)-4)
  
  routinename='setdefaultvalue'
  resolve_routine, routinename
  coyotepath=routine_info(routinename,/source)
  coyotepath=strmid(coyotepath.path,0,strlen(coyotepath.path)-strlen(routinename)-4)


  if not keyword_set(quiet) then print,format='("Generating scenes...")'

  ;Define time array
  timemin = 0. ;years
  if n_elements(timemax) eq 0 then timemax = 1e-10 ;years, by default this makes 1 timestep
  if n_elements(dt) eq 0 then dt = 10.0/365.25 ;years
  ntimes = ceil((timemax-timemin)/dt)
  time_vector = findgen(ntimes)/(ntimes-1) * (timemax-timemin) + timemin
  
  ;Define master wavelength array
  if n_elements(specres) eq 0 then specres = 300.
  if n_elements(lambdamin) eq 0 then lambdamin = 0.3         ;microns
  if n_elements(lambdamax) eq 0 then lambdamax = 1.0         ;microns
  dlnlambda = 1./specres
  lnlambdamax = alog(lambdamax)
  lnlambdamin = alog(lambdamin)
  nlambda = ceil((lnlambdamax-lnlambdamin)/dlnlambda)
  lambda = (findgen(nlambda))/(nlambda-1)*(lnlambdamax-lnlambdamin)+lnlambdamin
  lambda = exp(lambda)

  
  ;Define disk wavelength array
  ;If the disk image cube were saved at the same spectral resolution
  ;as the star, the files would be very large. We get around this by
  ;taking advantage of the fact that the dust's optical properties are
  ;smooth and don't have sharp features, and save the disk contrast
  ;instead of the flux. One can then interpolate this. By comparing
  ;a high res calculation of contrast (R=300) to an interpolated low-res
  ;I determined that R=10 was sufficient to resolve disk spectral features
  ;from 0.3-1 microns for a broad range of compositions (astrosil, waterice, organics, etc.)
  if n_elements(specrdisk) eq 0 then specrdisk = 10.
  dlnlambda = 1./specrdisk
  lnlambdamax = alog(lambdamax*1.01) ;we increase this a bit to make lambdamax_disk > lambdamax for interpolation later
  lnlambdamin = alog(lambdamin*0.99) ;we dedrease this a bit to make lambdamin_disk < lambdamin for interpolation later
  nlambda_disk = ceil((lnlambdamax-lnlambdamin)/dlnlambda)
  lambda_disk = (findgen(nlambda_disk))/(nlambda_disk-1)*(lnlambdamax-lnlambdamin)+lnlambdamin
  lambda_disk = exp(lambda_disk)

  
  if not keyword_set(quiet) then print,'  Adopting wavelengths from '+strcompress(string(lambdamin,format='(F0.2)'),/remove_all)+' - '+strcompress(string(lambdamax,format='(F0.2)'),/remove_all)+' microns with R = '+strcompress(string(specres,format='(F0.1)'),/remove_all)+' for star & planets, R = '+strcompress(string(specrdisk,format='(F0.1)'),/remove_all)+' for disks'


  
  ;Define debris disk grain sizes
  ;If we did Mie theory on the fly using dustmap, we'd have to use
  ;a LOT of different grain sizes to average over Mie ringing
  ;artifacts. Instead, we have pre-calculated Qabs and Qsca
  ;for different grain size ranges (in lqq_files folder).
  ;Here is the master list of pre-calculated grain sizes we can
  ;select from. I previously determined a size resolution ~ 5 was
  ;sufficient to accurately reproduce Qsca and Qabs for any distribution
  ;of grain sizes.
  sizeres = 5.
  master_maxsize = 1000.
  master_minsize = 0.1
  dlnsize = 1./sizeres
  master_lnmaxsize = alog(master_maxsize)
  master_lnminsize = alog(master_minsize)
  master_nsizes = ceil((master_lnmaxsize-master_lnminsize)/dlnsize)
  master_rdust = (findgen(master_nsizes)+0.5)/(master_nsizes)*(master_lnmaxsize-master_lnminsize)+master_lnminsize
  master_rdust = exp(master_rdust)
  master_rdust_boundaries = (findgen(master_nsizes+1))/(master_nsizes)*(master_lnmaxsize-master_lnminsize)+master_lnminsize
  master_rdust_boundaries = exp(master_rdust_boundaries)
  master_drdust = master_rdust_boundaries[1:master_nsizes]-master_rdust_boundaries[0:master_nsizes-1] ;********8
  lqq_dir = file_search('./lqq_files/',/fully_qualify_path)  
  lqq_dir = lqq_dir[0]+'/'



  
  ;Constants
  grav_const = 4.*!dpi*!dpi              ;in AU^3 yr^-2 solar_mass^-1
  pio180 = !dpi/180.
  pixscale_mas = pixscale_arcsec * 1000. ;mas
  nstars = n_elements(stars)
  ncomponents = n_elements(stars[0].disk)
  nplanets = n_elements(stars[0].planet)

  
  ;First we pre-load all geometric albedos 
  if not keyword_set(quiet) then print,format='("  Loading albedo files in their native spectral resolution...",$)'
  albedo_files = stars[*].planet[*].albedo_file
  albedo_files = albedo_files[uniq(albedo_files, sort(albedo_files))]
  albedo_files = albedo_files[where(albedo_files ne '')] ;list of unique albedo files
  nfiles = n_elements(albedo_files)


  nplambda = 0                  ;maximum number of wavelengths
  for ifile = 0, nfiles-1 do nplambda = file_lines(albedo_files[ifile]) > nplambda
  nplambda += (nlambda+1) ;we add on potentially nlambda+1 wavelengths (this becomes clear in a moment)
  transition_lambda = [(lambda[0]/lambda[1])*lambda[0],lambda,(lambda[nlambda-1]/lambda[nlambda-2])*lambda[nlambda-1]]
  transition_lambda = sqrt(transition_lambda[0:n_elements(transition_lambda)-2]*transition_lambda[1:n_elements(transition_lambda)-1])
  plambda = fltarr(nfiles,nplambda) 
  pgeometric_albedos = fltarr(nfiles,nplambda)
  for ifile = 0, nfiles-1 do begin
     d = read_ascii(albedo_files[ifile]) ;read the albedo file
     d = d.field1
     temp_lambda = reform(d[0,*]) ;first column of text file is wavelength in microns
     temp_inputg = reform(d[1,*]) ;second column is geometric albedo

     ;Get the geometric albedoes at the wavelengths half way
     ;between the master wavelengths. This allows us to bin 
     ;accurately later.
     transition_g = interpol(temp_inputg,temp_lambda,transition_lambda,/spline)
     transition_g = transition_g > 0. ;remove negative values

     ;Now combine, sort, and ask for unique values
     new_lambda = [temp_lambda,transition_lambda]
     new_g = [temp_inputg,transition_g]
     k = sort(new_lambda)
     new_lambda = new_lambda[k]
     new_g = new_g[k]
     k = uniq(new_lambda)
     new_lambda = new_lambda[k]
     new_g = new_g[k]

     ;Record in array
     plambda[ifile,0:n_elements(new_lambda)-1] = new_lambda
     pgeometric_albedos[ifile,0:n_elements(new_lambda)-1] = new_g
  endfor     
  if not keyword_set(quiet) then print,'done.'

  
  ;Loop through each star and make images
  bksp = ''

  ;save all data to a save file for easy access by threads
  exchangefile = exovistapath+'multithread_exchange_file.sav'
  save,filename=exchangefile    ;save 

  obridge=obj_new("IDL_IDLBridge")             ; Initiate IDL bridge/child process
  obridge = replicate(obridge,nthreads)          ;make an array of child processes
  status = intarr(nthreads)                      ;keeps track of each child process' status

  nstarsperthread = ceil((nstars*1.0)/nthreads)
  print,'nstarsperthread = ',nstarsperthread
  for ithread=0, nthreads-1 do begin

     istarmin = ithread*nstarsperthread
     istarmax = ((ithread+1)*nstarsperthread-1) < (nstars-1)
     
     ;Update the user on progress...
     if not keyword_set(quiet) then print, '  Thread ' + strcompress(string(ithread+1,format='(I0)'),/remove_all) + ' / ' + strcompress(string(nthreads,format='(I0)'),/remove_all)+': ',istarmin,istarmax


     
     obridge[ithread]=obj_new("IDL_IDLBridge", output=output_dir+string(ithread+1,format='(I0)')+'.txt') ; Initiate IDL bridge/child process
     exstring="!PATH = '" + exovistapath + ":' + !PATH"
     obridge[ithread]->execute,exstring
     exstring="!PATH = '" + astronpath + ":' + !PATH"
     obridge[ithread]->execute,exstring
     exstring="!PATH = '" + coyotepath + ":' + !PATH"
     obridge[ithread]->execute,exstring
     obridge[ithread]->setvar, 'exchangefile', exchangefile
     obridge[ithread]->setvar, 'istarmin', istarmin
     obridge[ithread]->setvar, 'istarmax', istarmax
     obridge[ithread]->execute, 'thread_the_scene,exchangefile,istarmin,istarmax', /nowait ; Run disk_dustmap and continue

  endfor
  
  nrunning = nthreads           ;make sure we enter the loop the first time
  ;status = 0 means not started yet or finished
  ;status = 1 means running
  ;status = 2 means finished
  while nrunning gt 0 do begin

     ;Update the status of any unfinished threads
     for j=0,nthreads-1 do status[j] = obridge[j]->status(error=errorstring)
     print,systime(),' - STATUS = ',string(status,format='(I0)')

     ;Figure out how many are running
     nrunning = 0
     for j=0,nthreads-1 do begin
        if status[j] eq 1 then nrunning++
     endfor
        
     wait,10.0 ;wait before checking again
  endwhile

  print,' '
  print,'  ...done.'

  ;Delete any running processes and clean up
  for j=0,nthreads-1 do begin
     obridge[j]->Abort
     obridge[j]->Cleanup
  endfor
  
  ;reset the seed so as to avoid overwriting it
  seed = input_seed

end

