pro remove_unstable_planets, stars, hillsphere_flag

  nstars = n_elements(stars)
  mhs_factor = 6 ;Dulz et al use 9, but that takes a really
                 ;long time to cram them in. So we speed things
                 ;up by relaxing this.

  for i=0, nstars-1 do begin

     if hillsphere_flag[i] then begin
        temp_planets = stars[i].planet[*]
        threestarmass_EM = 3 * stars[i].mass * 333000. ;star mass in Earth masses
        ;They have been sorted by semi-major axis, so now look
        ;at them pairwise
        j = 0
        while temp_planets[j+1].r gt 0 do begin
           aout = temp_planets[j+1].a
           ain = temp_planets[j].a
           Mout = temp_planets[j+1].m ;Earth masses
           Min = temp_planets[j].m    ;Earth masses
           mutual_hill_sphere = 2 * (aout-ain)/(aout+ain) * (threestarmass_EM / (Mout + Min))^(0.33333333)
           if mutual_hill_sphere lt mhs_factor then begin
                                ;erase the less massive planet
              if Mout gt Min then j0 = j else j0 = j+1
              temp_planets[j0].M = 0.
              temp_planets[j0].r = 0.
              temp_planets[j0].albedo_file = ''
              temp_planets[j0].a = 1e20
              temp_planets[j0].e = 0.
              temp_planets[j0].i = 0.
              temp_planets[j0].longnode = 0.
              temp_planets[j0].argperi = 0.
              temp_planets[j0].meananom = 0.
              
              l = sort(temp_planets.a) ;re-sort to get rid of deleted planet
              temp_planets = temp_planets[l]
                                ;do not increase j if a planet was deleted
           endif else begin
              j++               ;if the planet pair was okay, increase j
           endelse
        endwhile
        
                                ;We have finished deleting the unstable planets for this star
                                ;Now record the surviving planets
        stars[i].planet[*] = temp_planets[*]
        
        hillsphere_flag[i] = 0
     endif
  endfor
     
     
end


pro add_planets, stars, expected, occurrence_rates, ora_array, orM_array, occrate_lnMmin, occrate_lnamin, occrate_dlnM, occrate_dlna, npix, emin, emax, imin, imax, hillsphere_flag, seed=seed


  nstars = n_elements(stars)
  planet_entry = stars[0].planet[0]
  maxnplanets = n_elements(stars[0].planet[*].a)
  hillsphere_flag[*] = 0 ;reset this flag to no recalculation for all stars
  
  ;First make a histogram of what we have
  tempstars = stars ;first make a copy
  for i=0,nstars-1 do begin
     tempstars[i].planet[*].a /= sqrt(stars[i].Lstar) ;undo the sqrt(L) factor
  endfor
  temp_a = tempstars[*].planet[*].a
  temp_M = tempstars[*].planet[*].M

  j = where(temp_M gt 0)
  ;make the histo
  if j[0] ne -1 then begin
     h = hist_2d(alog(temp_a[j]),alog(temp_M[j]),bin1=occrate_dlna,bin2=occrate_dlnM,min1=occrate_lnamin,min2=occrate_lnMmin,max1=occrate_lnamin+occrate_dlna*npix,max2=occrate_lnMmin+occrate_dlnM*npix)
     h = h[0:npix-1,0:npix-1]
  endif else begin
     h = lonarr(npix,npix) ;if there are no planets (first time through)
  endelse


  
  ;loadct,39
  ;tvscl,expected,0,0
  ;tvscl,h,npix,0



  ;Now create the planets that we need
  nplanets_histo = expected - h ;2d histogram of # of needed planets
  ;Efficiently create indices for all planets that need to be created
  j = where(nplanets_histo gt 0)
  for i=0,n_elements(j)-1 do begin
     if n_elements(indices) eq 0 then begin
        indices = replicate(j[i],nplanets_histo[j[i]])
     endif else begin
        indices = [indices,replicate(j[i],nplanets_histo[j[i]])]
     endelse
  endfor
  nplanets = n_elements(indices)
  ;For each of those planets, calculate parameters
  temp_planets = replicate(planet_entry,nplanets)
  ;Randomize some angles (this could result in unstable systems)
  temp_planets.longnode = randomu(seed,nplanets)*360.
  temp_planets.argperi = randomu(seed,nplanets)*360.
  temp_planets.meananom = randomu(seed,nplanets)*360.
  ;Randomize eccentricity (this could result in unstable systems)
  temp_planets.e = emin + randomu(seed,nplanets)*(emax-emin)
  ;Randomize inclination to within some tolerance of the system
  temp_planets.i = (imin + randomu(seed,nplanets)*(imax-imin))
  ;Randomly pick a mass within that cell of the occurrence rates
  temp_planets.M = exp((randomu(seed,nplanets))*occrate_dlnm)*orM_array[indices]
  ;Randomly pick a semi-major axis within that cell of the occurrence rates
  temp_planets.a = exp((randomu(seed,nplanets))*occrate_dlna)*ora_array[indices]
  ;Calculate radius in Earth radii based on Chen & Kipping (2017)
  temp_planets.r = mass_to_radius(temp_planets.m) ;Calculate radius in Earth radii based on Chen & Kipping (2017)


  
  ;Randomly assign to stars
  starindices = floor(randomu(seed, nplanets) * nstars)  
  for iplanet=0,nplanets-1 do begin
     starindex = starindices[iplanet]
     the_planet = temp_planets[iplanet]
     the_planet.a *= sqrt(stars[starindex].Lstar) ;scale by sqrt(Lstar)
     new_planets = stars[starindex].planet
     new_planets[maxnplanets-1] = the_planet ;put the new planet in the last planet entry
     new_planets = new_planets[sort(new_planets.a)] ;sort by semi-major axis
     stars[starindex].planet = new_planets
     hillsphere_flag[starindex] = 1 ;any star w/ a planet added needs recalculation of hill spheres
  endfor

  
end

pro generate_planets, stars, ul=ul, ll=ll, emin=emin, emax=emax, imin=imin, imax=imax, sysimin=sysimin, sysimax=sysimax, sysPAmin=sysPAmin, sysPAmax=sysPAmax, seed=seed, npix=npix, quiet=quiet, nomcdraw=nomcdraw

  ;ul = uses the upper limit of the occurrence rates
  ;ll = uses the lower limit of the occurrence rates 
  ;emin = lower limit on uniform eccentricity distribution
  ;emax = upper limit on uniform eccentricity distribution
  ;imin = lower limit on uniform inclination distribution (deg)
  ;imax = upper limit on uniform inclination distribution (deg)
  ;sysimin = lower limit on midplane inclination distribution (deg)
  ;sysimax = upper limit on midplane inclination distribution (deg)
  ;sysPAmin = lower limit on midplane PA distribution (deg)
  ;sysPAmax = upper limit on midplane PA distribution (deg)
  ;npix = sets the occurrence rate resolution (min of 100 is suggested, max of 1000)
  ;quiet = turn off all status prints
  ;nomcdraw = if set, don't do a Monte Carlo draw on occ. rates--use expected value instead  


  ;define some variables
  pio180 = !dpi/180.
  twopi = 2*!dpi
  if n_elements(npix) eq 0 then npix = 100
  if npix lt 100 then begin
     print,'Warning: npix < 100 is not recommended.'
     pause
  endif
  if n_elements(seed) eq 0 then seed = systime(/sec) ;by default the results are not reproducible
  input_seed = seed ;save this seed
  if n_elements(emin) eq 0 then emin = 0.
  if n_elements(emax) eq 0 then emax = 0.
  if emax lt emin then stop,'Error. emax must be greater than emin.'
  if n_elements(imin) eq 0 then imin = 0.
  if n_elements(imax) eq 0 then imax = 5. ;5 degrees by default
  if imax lt imin then stop,'Error. imax must be greater than imin.'  

  if n_elements(sysimin) eq 0 then sysimin = 0. ;deg
  if n_elements(sysimax) eq 0 then sysimax = 180. ;deg
  if n_elements(sysPAmin) eq 0 then sysPAmin = 0. ;deg
  if n_elements(sysPAmax) eq 0 then sysPAmax = 360. ;deg
  cossysimin = cos(sysimin*pio180)
  cossysimax = cos(sysimax*pio180)

  nstars = n_elements(stars)

  ;Define planet structure entry, as described below
  if not keyword_set(quiet) then print,format='("Generating planets...")'
  maxnplanets = 30
  planet_entry = {M:0.0, R:0.0, albedo_file:'', a:0.0, e:0.0, i:0.0, longnode:0.0, argperi:0.0, meananom:0.0}
  planet_entries = replicate(planet_entry,maxnplanets)
  ;M: mass of planet in Earth masses
  ;R: radius of planet in Earth radii
  ;albedo_file: relative path to geometric albedo file
  ;a: semi-major axis (physical, not projected) in AU
  ;e: eccentricity (physical, not projected)
  ;i: inclination relative to system midplane
  ;longnode: longitude of the ascending node
  ;argperi: arg. of pericenter
  ;meananom: mean anomaly of the planet at t=0
  
  ;Check that "stars" has the necessary tags
  if not tag_exist(stars,'LSTAR',/quiet) then stop,'stars input must be a structure with tagname LSTAR'

  ;We modify the input structure to include planets (if it hasn't already been done)
  j = where(tag_names(stars) eq 'PLANET')
  if j[0] ne -1 then begin
     if array_equal(tag_names(planet_entries), tag_names(stars.planet)) then begin
        print,"'planet' already present in input structure. Erasing and regenerating..."
        stars[*].planet = planet_entries
     endif else begin
        print,"'planet' tag already exists in input structure but is not of expected format. Remove 'planet' tag from input structure. "
        stop
     endelse
  endif
  if j[0] eq -1 then begin  ;if the structure needs planets added...
     temp = replicate(create_struct('planet',planet_entries,'PA',0.,'I',0.,stars[0]),nstars) ;make new array of structures w/ planet tag
     struct_assign,stars,temp ;place values from stars array structure into temp
     stars = temp ;overwrite
  endif     
  stars[*].planet[*].a = 1e20 ;we set semi-major axis for non-existent planets to this for sorting purposes
  
  ;Randomly assign system midplane orientation
  stars.PA = (randomu(seed,nstars)*(sysPAmax-sysPAmin)+sysPAmin)
  stars.i = acos(randomu(seed,nstars)*(cossysimin-cossysimax) + cossysimax)/pio180
  temprand = randomu(seed,nstars)
  j = where(temprand lt 0.5)
  if j[0] ne -1 then stars[j].i -= 180.

  
  ;load occurrence rates in mass - semimajor axis space
  load_occurrence_rates, occrates, occrate_lnMmin, occrate_lnamin, occrate_dlnM, occrate_dlna, npix, a_array=ora_array, r_array=orM_array, ul=ul, ll=ll,/mass,quiet=quiet ;a is along x-axis, M is along y-axis
  orr_array = mass_to_radius(orM_array)   ;Calculate radius in Earth radii based on Chen & Kipping (2017)
  expected_planets_per_star = total(occrates)

  
  ;Figure out how many planets we should have in the end.
  ;To get "exactly" the right number of planets per star we'd do this:
  ;expected = round(occrates * nstars)
  ;But it fails if nstars is small or occurrence rates is small in a bin.
  ;So instead, we do a monte carlo draw and the # of planets
  ;may not be exactly equal to the occurrence rate prediction.
  if keyword_set(nomcdraw) then begin ;Don't do the Monte Carlot draw method
     expected = round(occrates*nstars)
  endif else begin ;normally do the Monte Carlo draw method
     expected = lonarr(npix,npix)
     for i=0,nstars-1 do begin
        mcdraw = randomu(seed,npix,npix)
        j = where(mcdraw lt occrates)
        if j[0] ne -1 then expected[j]++ 
     endfor
  endelse
  nexpected = total(expected)
  
  hillsphere_flag = intarr(nstars) ;improves run time efficiency: 0=don't need to calculate, 1=calculate
  nplanets = 0 ;make sure we enter the loop
  n_nochange = 0
  while nplanets lt nexpected and n_nochange lt 50 do begin

     prev_nplanets = nplanets
     
     ;First, add some random planets based on occurrenc rates 
     add_planets, stars, expected, occrates, ora_array, orM_array, occrate_lnMmin, occrate_lnamin, occrate_dlnM, occrate_dlna, npix, emin, emax, imin, imax, hillsphere_flag, seed=seed

     ;Now remove any unstable ones
     remove_unstable_planets, stars, hillsphere_flag

     ;How many planets do we have now?
     nplanets = 0
     j = where(stars[*].planet[*].r gt 0)
     if j[0] ne -1 then nplanets = n_elements(j)

     dnp = nplanets-prev_nplanets
     if dnp eq 0 then n_nochange++ else n_nochange = 0 ;if it gets stuck then we need to break out
     
  endwhile


  ;Erase semi-major axis of non-existant planets
  for i=0, nstars-1 do begin
     j = where(stars[i].planet[*].r eq 0)
     if j[0] ne -1 then stars[i].planet[j].a = 0.0
  endfor
  
  
  ;How many planets were made?
  j = where(stars[*].planet[*].R ne 0)
  if not keyword_set(quiet) then begin
     print,'  '+strcompress(string(total(occrates)*nstars,format='(F0.1)'),/remove_all)+' planets expected from occurrence rates'
     if not keyword_set(nomcdraw) then print,'  '+strcompress(string(nexpected,format='(F0.1)'),/remove_all)+' planets expected from Monte Carlo draw of occurrence rates'
     print,'  '+strcompress(string(n_elements(j),format='(I0)'),/remove_all)+' planets generated after imposing stability limits.'
  endif

    
  ;Randomly assign all planets an albedo file
  print,'  Assigning planet types/albedos based on rules in assign_albedo_file.pro.'
  assign_albedo_file, stars, ora_array, orr_array, orM_array, seed=seed

  
  ;reset the seed so as to avoid overwriting it
  seed = input_seed

  print,'  ...done.'

end
