pro generate_disks, stars, imin=imin, imax=imax, PAmin=PAmin, PAmax=PAmax, seed=seed, noretrograde=noretrograde, ncomponents=ncomponents, quiet=quiet

  
  ;INPUTS
  ;stars: array of star structure (e.g., from load_stars.pro)
  ;planets: array of planet structure (from generate_planets.pro)
  ;imin: min inclination in deg of disk on sky, default=0
  ;imax: max inclination in deg of disk on sky, default=180
  ;PAmin: min PA in deg of disk on sky (E of N), default=0
  ;PAmax: max PA in deg of disk on sky (E of N), default=360
  ;noretrograde: set this to ensure all systems have positive inclination
  ;ncomponents: the # of disk components to generate (2 by default)
  
  ;Define a disk structure entry, as described below
  nstars = n_elements(stars)
  if n_elements(ncomponents) eq 0 then ncomponents = 2               ;# of disk components to model
  if ncomponents eq 0 or ncomponents gt 3 then stop,'ERROR: Must choose 1-3 disk components'
  disk_entry = {longnode:double(0.0), i:double(0.0), nzodis:double(0.), r:double(0.), dror:double(0.), rinner:double(0.), eta:double(0.), hor:double(0.), g:dblarr(3), w:dblarr(3)}
  disk_entries = replicate(disk_entry,ncomponents)
  ;i: inclination of disk relative to midplane
  ;longnode: longitude of ascending node of disk
  ;nzodis: density of each disk component (3 components max)
  ;r: radial location of each dust belt component
  ;dror: width of belt
  ;rinner: inner truncation radius of each component
  ;eta: PR drag time divided by collision time scale for blowout size
  ;hor: scale height of each component
  ; NOTE: each disk component is assigned a linear combination of
  ; 3 Henyey-Greenstein scattering phase functions.
  ;g: HG asymmetry parameters (3 values for each component)
  ;w: HG weighting factors for each function and component

  
  ;Define some variables
  pio180 = !dpi/180.
  twopi = 2*!dpi
  density_ratio = 5.            ;all components have the same density w/in this factor
  stability_factor = 3.
  rinner_mass_threshold = 100.
  dror_min = 0.05 ;dust belts must be at least this wide
  dror_max = 0.3
  hor_min = 0.03
  hor_max = 0.2
  r_min = [0.5,5.,50.] ;min circumstellar distance of each disk component
  r_max = [5.0,50.,500.] ;max circumstellar distance of each disk component
  grav_const = 4*!dpi*!dpi               ;AU^3 yr^-2 MSun^-1
  c = 2.998e8 / 1.496e11 * 365.25 * 24. * 60. * 60. ;AU yr^-1  
  
  if n_elements(seed) eq 0 then seed = systime(/sec) ;by default the results are not reproducible
  input_seed = seed ;save this seed


  ;Check that "stars" has the necessary tags *****************************
  if not tag_exist(stars,'LSTAR',/quiet) then stop,'stars input must be a structure with tagname LSTAR'
  if not tag_exist(stars,'MASS',/quiet) then stop,'stars input must be a structure with tagname MASS'
  if not tag_exist(stars,'PA',/quiet) then stop,'stars input must be a structure with tagname PA'
  if not tag_exist(stars,'I',/quiet) then stop,'stars input must be a structure with tagname I'
  if not tag_exist(stars,'PLANET',/quiet) then stop,'stars input must be a structure with tagname PLANET'

  if not keyword_set(quiet) then print,format='("Generating ", I0 ," disk components per star...",$)',strcompress(string(ncomponents,format='(I0)'),/remove_all)
  
  ;We modify the input structure to include disks (if it hasn't already been done)
  j = where(tag_names(stars) eq 'DISK')
  if j[0] ne -1 then begin
     if array_equal(tag_names(disk_entries), tag_names(stars.disk)) then begin
        print,"'disk' already present in input structure. Erasing and regenerating..."
        stars[*].disk = disk_entries
     endif else begin
        print,"'disk' tag already exists in input structure but is not of expected format. Remove 'disk' tag from input structure. "
        stop
     endelse
  endif
  if j[0] eq -1 then begin  ;if the structure needs disks added...
     temp = replicate(create_struct('disk',disk_entries,stars[0]),nstars) ;make new array of structures w/ planet tag
     struct_assign,stars,temp ;place values from stars array structure into temp
     stars = temp ;overwrite
  endif     

  ;all disks assigned the same orientation as system midplane
  for icomp=0, ncomponents-1 do begin
     stars.disk[icomp].longnode = 0
     stars.disk[icomp].i = 0
  endfor
  
  ;Distribute exozodi levels
  nexozodi_levels = readfits('./LBTI_exozodi_distributions/nominal_maxL_distribution-Dec2019.fits',h)
  nexozodi_levels = shuffle(nexozodi_levels,seed=seed)
  stars.disk[0].nzodis = nexozodi_levels[0:nstars-1]  
  maxfac = alog10(density_ratio) ;outer disks must be within some density factor of inner disk
  minfac = alog10(1./density_ratio)
  if ncomponents gt 1 then begin
     for icomp=1, ncomponents-1 do begin
        stars.disk[icomp].nzodis = 10.^(randomu(seed,nstars)*(maxfac-minfac)+minfac)*stars.disk[0].nzodis
     endfor
  endif
  

  
  ;Define spatial distribution of dust
  ;We do this on a star by star basis
  ;This allows us to generate rules based on the underlying planetary system
  for i=0, nstars-1 do begin

     s = stars[i]     
     p = s.planet
     nplanets = n_elements(p)
     
     ;generate logarithmically spaced circumstellar distance vector
     minr = min(r_min)
     maxr = max(r_max)
     npoints = 10000
     r = findgen(npoints)/(npoints-1.)*(alog10(maxr)-alog10(minr))+alog10(minr)
     r = 10.^r ;circumstellar distances to consider
     valid = r*0+1 ;valid circumstellar distances for dust to exist

     ;Enforce Hill sphere stability rules
     ;Calculate Hill radii of all planets
     ;Note that all missing planets will have a hill radius = 0
     hill_radius = p.a * (1.-p.e) * (p.M / (3 * s.MASS * 333000.))^(1./3.) ;AU

     ;Determine where we can put dust
     ;Rule out unstable regions
     ;Note this works for missing planets because hill radius = 0
     for ip=0,nplanets-1 do begin
        if p[ip].R eq 0 then break ;stop if we reach missing planets
        j = where(abs(r-p[ip].a) lt stability_factor*hill_radius[ip])
        if j[0] ne -1 then valid[j] = 0
     endfor
        
     ;For each stable region, record largest possible dr
     dr = dblarr(npoints)
     for ir=0,npoints-1 do begin
        if valid[ir] eq 1 then begin
           minr = r[ir]
           irmin = ir
           while valid[ir] eq 1 and ir lt npoints-1 do ir++
           maxr = r[ir]
           irmax = ir
           dr[irmin:irmax] = abs(maxr-r[irmin:irmax]) < abs(minr-r[irmin:irmax]) ;the largest dr possible at each valid r
        endif
     endfor
     
     ;Ignore the really narrow belts
     dror = dr/r
     j = where(dror lt dror_min)
     if j[0] ne -1 then valid[j] = 0
     
        
     ;Now we know where we can put the dust
     ;Let's distribute each component
     ;First, the easy stuff that doesn't depend on valid...
     stars[i].disk[*].hor = randomu(seed,ncomponents)*(hor_max - hor_min) + hor_min
     stars[i].disk[*].g[0] = randomu(seed,ncomponents)*(0.995-0.8)+0.8        ;most forward scatt term
     stars[i].disk[*].g[1] = randomu(seed,ncomponents)*(0.8-0.35)+0.35 ;medium forward scatt term
     stars[i].disk[*].g[2] = randomu(seed,ncomponents)*(0.35+0.3)-0.3 ;least forward scatt term
     stars[i].disk[*].w[0] = randomu(seed,ncomponents,/double)*(0.8-0.4)+0.4 ;most forward scatt term
     minw = double(0.1)
     maxw = double(1.0)-stars[i].disk[*].w[0]
     stars[i].disk[*].w[1] = randomu(seed,ncomponents,/double)*(maxw-minw)+minw ;medium forward scatt term
     stars[i].disk[*].w[2] = double(1.0) - stars[i].disk[*].w[0] - stars[i].disk[*].w[1] ;least forward scatt term


     ;stars[i].disk[*].g[0] = randomu(seed,ncomponents)*(0.98-0.5)+0.5        ;most forward scatt term
     ;stars[i].disk[*].g[1] = randomu(seed,ncomponents)*(0.4-0.2)+0.2 ;medium forward scatt term
     ;stars[i].disk[*].g[2] = randomu(seed,ncomponents)*(0.2+0.2)-0.2 ;least forward scatt term
     ;stars[i].disk[*].w[0] = randomu(seed,ncomponents,/double)*(0.8-0.4)+0.4 ;most forward scatt term
     ;minw = double(0.1)
     ;maxw = double(1.0)-stars[i].disk[*].w[0]
     ;stars[i].disk[*].w[1] = randomu(seed,ncomponents,/double)*(maxw-minw)+minw ;medium forward scatt term
     ;stars[i].disk[*].w[2] = double(1.0) - stars[i].disk[*].w[0] - stars[i].disk[*].w[1] ;least forward scatt term
     
     ;now the more complicated part...
     for icomp = 0, ncomponents-1 do begin

        ;If we were to pick the belt based on max(dr/r), i.e. region
        ;of maximum stability, we would tend to get belts that are
        ;all near r_max[icomp]. So this method is abandoned:
        ;tempvalid = valid       ;create a copy of valid
        ;j = where(r lt r_min[icomp] or r gt r_max[icomp])
        ;if j[0] ne -1 then tempvalid[j] = 0 ;remove locations beyond our region of interest
        ;maxdror = max(dr/r*tempvalid,maxloc) ;find location of max(dr/r) that is also valid
        ;if maxdror gt dror_min then begin    ;cannot be a super thin region
        ;   stars[i].disk[icomp].r = r[maxloc]
        ;   stars[i].disk[icomp].dror = randomu(seed,1)*((maxdror < dror_max) - dror_min) + dror_min
        ;   stars[i].disk[icomp].rinner = 0.
        ;   ;Determine if a massive planet could truncate the inward migrating component
        ;   j = where(p.a lt stars[i].disk[icomp].r and p.M gt rinner_mass_threshold)
        ;   if j[0] ne -1 then stars[i].disk[icomp].rinner = (1.1 * max(p[j].a * (1+p[j].e))) < stars[i].disk[icomp].r
        ;endif

        ;Instead, we randomly draw a location from the
        ;logarithmically-spaced valid r values...
        j = where(valid eq 1 and r gt r_min[icomp] and r lt r_max[icomp]) ;get the possible belt locations
        if j[0] ne -1 then begin
           randindex = randomu(seed,1)*n_elements(j)
           stars[i].disk[icomp].r = r[j[randindex]]
           stars[i].disk[icomp].dror = dror[j[randindex]]
           stars[i].disk[icomp].rinner = 0.0
           ;Determine if a massive planet could truncate the inward migrating component
           j = where(p.a lt stars[i].disk[icomp].r and p.M gt rinner_mass_threshold)
           if j[0] ne -1 then stars[i].disk[icomp].rinner = (1.1 * max(p[j].a * (1+p[j].e))) < stars[i].disk[icomp].r
        endif else begin
           stars[i].disk[icomp].nzodis = 0.0 ;couldn't find a valid location, so zero it out
        endelse
     endfor


     ;Now that we have the locations we can determine collision rates...
     for icomp = 0, ncomponents-1 do begin        
        beta = 0.5
        alpha = beta * grav_const * s.MASS / c ;quantity related to PR drag to be used later
        tpr = stars[i].disk[icomp].r^2 / (4 * alpha) ;yr
        torbit = sqrt(stars[i].disk[icomp].r^3 / s.MASS) ;yr
        tcoll = torbit / (4*!dpi*stars[i].disk[icomp].nzodis*1e-7) ;yr
        stars[i].disk[icomp].eta = tpr / tcoll
     endfor

  endfor

  
  
  ;reset the seed so as to avoid overwriting it
  seed = input_seed

  if not keyword_set(quiet) then print,'done.'
end
