function disk_imager, s, x, y, z, r, dv, cosscattang, rdust, drdust, rdust_blowout, Qsca, tsublimate

  ;d is a structure containing all disk information
  ;x is an array of x coordinates for points in the disk plane
  ;y is an array of y coordinates for points in the disk plane
  ;z is an array of z coordinates for points in the disk plane
  ;dv is the differential volume represented by each point
  ;cosscattang: cosine of the scattering angle for all points
  ;rdust: grain size vector (microns)
  ;drdust: grain bin size (microns)
  ;Qsca: scattering cross section vs lambda
  
  dims = size(x,/dim)
  nsizes = n_elements(rdust)
  nlambda = size(Qsca,/dim)
  if n_elements(nlambda) eq 1 then nlambda = 1 else nlambda = nlambda[1]

  ;Here is the normalization factor that gives the proper
  ;V band surface brightness for 1 zodi solar system twin.
  ;I play pretty fast and loose with units in this routine, 
  ;so this includes all of the unit conversions that I skip.
  n0 = 6.6e-08            ;normalization 


  flux = dblarr(dims[0],dims[1],nlambda) ;the returned flux will be resolved by subpixel
  r2 = r*r

  ;Calculate grain temp. For efficiency we don't do this iteratively.
  ;We just use BB temp and assume an albedo of 0.5.
  ;Also adopt a 1.5x fudge factor for grain super-heating.
  rstarAU = s.rstar * 0.00465047
  t = s.teff * sqrt((sqrt(1.0-0.5) * rstarAU) / (2*r)) * 1.5 ;1.5x factor is a fudge factor for grain super-heating

  for i=0,n_elements(s.disk)-1 do begin
     d = s.disk[i]

     if d.nzodis gt 0 then begin
     
        r0 = double(d.r)
        dr = double(d.dror * r0)
        oneodr = double(1.0/dr)
        oneor0pdr = double(1.0/(r0+dr))
        oneor0mdr = double(1.0/(r0-dr))

        
        ;Calculate the scattering phase function for this disk component (multi-component HG)
        pfunc = dblarr(dims)
        for ihg = 0, n_elements(d.w)-1 do begin
           w = d.w[ihg]
           g = d.g[ihg]
           temp = (double(1.0) + g*g) - (double(2.0)*g)*cosscattang
           pfunc += (w * (double(1.0)-g*g)/(double(4.0)*!dpi)) / sqrt(temp*temp*temp)
        endfor

        
        ;Now calculate disk density at each point
        n = dblarr(dims)        ;all points set to zero
        j = where(t lt tsublimate and r gt rstarAU)
        if j[0] ne -1 then n[j] = n0*d.nzodis ;those outside of sublimation radius set to nzodis

     
        ;Multiply by differential volume
        n *= dv

     
        ;Modify by the variation w/ z
        n *= (exp(double(-0.5) * (z/(d.hor * r))^2)) 
        n /= ((d.hor * sqrt(2 * !dpi)) * r) ;normalize by the integral over z to maintain same density vs r
        
        ;The Gaussian ring component
        j = where(abs(r-r0) lt dr)
        if j[0] ne -1 then n[j] *= exp(double(-0.5) * ((r[j]-r0)*oneodr)^2)
        
     
        ;The blowout component
        ;Make sure it's continuous at +dr from r0
        j = where(r gt (r0+dr))
        if j[0] ne -1 then begin
           temp = r[j] * oneor0pdr
           n[j] *= (exp(double(-0.5))) / sqrt(temp * temp * temp)
        endif


        ;The inner truncation radius
        j = where(r le d.rinner)
        if j[0] ne -1 then begin
           rorinner = r/d.rinner
           n[j] *= rorinner[j]^3
        endif


        ;Multiply by phase function and 1/r^2     
        n *= (pfunc/((double(4.0)*!dpi)*r2))

        
        ;Do some calculations used in the loop outside of the loop
        j = where(r lt (r0-dr))
        temp = (double(1.0) - sqrt(r * oneor0mdr))   
        grainfactors = !dpi*rdust^(double(-3.5+2.0))*drdust
        for isize = 0, nsizes-1 do begin

           tempn = n*grainfactors[isize] ;make a copy and x by Donanyi size dist and cross section
           
           tempeta = double(d.eta * (rdust[isize]/rdust_blowout)) ;this creates a radial color gradient in disk
        
           ;Finally take care of the inward decay component
           ;It can change depending on grain size
           ;Make sure it's continuous at -dr from r0
           if j[0] ne -1 then tempn[j] *= (exp(double(-0.5))) / (double(1.0) + (double(4.0) * tempeta) * temp[j])

        
           k = where(finite(tempn) ne 1)
           if k[0] ne -1 then stop,'tempn NaN!'
           k = where(finite(tempn) eq 0)
           if k[0] ne -1 then stop,'tempn 0!'
  
        
           ;Now we can integrate along z-axis
           tempflux = total(tempn,3) ;it's now a 2D array

           
           ;Now we can rebin to include wavelength dependence 
           tempflux = rebin(tempflux,dims[0],dims[1],nlambda)
           tempQsca = dblarr(1,1,nlambda,/nozero) 
           tempQsca[0,0,*] = reform(Qsca[isize,*])
           tempQsca = rebin(tempQsca,dims[0],dims[1],nlambda)
           tempflux *= tempQsca ;include Qsca wavelength dependence for this grain size
           
           flux += tempflux
        endfor
     endif
  endfor

  return, flux
  
end
