pro assign_albedo_file, stars, a_array, r_array, M_array, seed=seed

  ;Here we assign random albedo files to the grid of planets
  ;Note that this is currently designed to only do this in 
  ;stellar-flux space, i.e. models are selected based on
  ;stellar radiation received from planet, not absolute 
  ;semi-major axis
  
  
  if n_elements(seed) eq 0 then seed = 0
  nstars = n_elements(stars)
  directory = './geometric_albedo_files/'
  xpix = size(a_array,/dim)
  ypix = xpix[1]
  xpix = xpix[0]

  ;Determine the bounds and resolution of the occurrence rate map
  dlna = alog(a_array[1:xpix-1,0])-alog(a_array[0:xpix-2,0])
  if abs(min(dlna)-max(dlna))/abs(min(dlna)) gt 0.01 then stop,'a_array must have uniform ln spacing'
  dlnM = alog(M_array[0,1:ypix-1])-alog(M_array[0,0:ypix-2])
  if abs(min(dlnM)-max(dlnM))/abs(min(dlnM)) gt 0.01 then stop,'M_array must have uniform ln spacing'
  dlna = dlna[0]
  dlnM = dlnM[0]

  minlna = min(alog(a_array))
  maxlna = max(alog(a_array)) + dlna
  minlnM = min(alog(M_array))
  maxlnM = max(alog(M_array)) + dlnM
  
  
  
  ;To add a planet, place the file name in the list below
  ;(order doesn't matter). Then define the rules below
  files = ['Archean_Earth', 'Earth', 'Hazy_Archean_Earth', $
           'Proterozoic_Earth-hi_o2', 'Proterozoic_Earth-lo_o2', $
           'Venus', 'Mars', 'Early_Mars', $ 
           'Jupiter', 'Saturn', 'Uranus', 'Neptune', $
           'Warm_Neptune_1AU_Clouds', 'Warm_Neptune_1AU_NoClouds', 'Warm_Neptune_2AU']
  nfiles = n_elements(files)
  
  ;Here are the rules
  valid = intarr(xpix,ypix,nfiles*100) ;100x nfiles lets us assign probabilities to the nearest percent
  valid[*,*,*] = -1 ;set to -1 by default (important later)
  
  EEC = r_array*0 ;this is going to mark the phase space of the exoEarth candidates (EECs)
  j = where(r_array gt 0.8/sqrt(a_array) and r_array lt 1.4 and a_array gt 0.95 and a_array lt 1.67)  ;EEC phase space rule
  EEC[j] = 1 ;mark the EEC phase space
  
  ;-------------------
  ;EEC STUFF FIRST
  k = where(files eq 'Archean_Earth')  ;albedo file name
  j = where(EEC eq 1) ;the rule to find valid phase space
  prob = 39 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Earth')
  j = where(EEC eq 1)
  prob = 24 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Hazy_Archean_Earth')
  j = where(EEC eq 1)
  prob = 5 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif

  k = where(files eq 'Proterozoic_Earth-hi_o2')
  j = where(EEC eq 1)
  prob = 16 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif

  k = where(files eq 'Proterozoic_Earth-lo_o2')
  prob = 16 ;relative weighting of all planets in this phase space (out of 100)
  j = where(EEC eq 1)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  ;END OF EEC
  ;---------------


  
  k = where(files eq 'Venus') ;albedo file
  j = where(r_array gt 0.7 and r_array lt 1.5 and a_array gt 0. and a_array lt 1. and EEC ne 1)  ;the rule to find valid phase space
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Mars')
  j = where(r_array gt 0.0 and r_array lt 2.0 and a_array gt 0.0 and a_array lt 1e20 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif

  k = where(files eq 'Early_Mars')
  j = where(r_array gt 0.3 and r_array lt 2.0 and a_array gt 0.95 and a_array lt 2 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif

  k = where(files eq 'Jupiter')
  j = where(r_array gt 6. and r_array lt 1e20 and a_array gt 0.0 and a_array lt 1e20 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif

  k = where(files eq 'Saturn')
  j = where(r_array gt 6. and r_array lt 10. and a_array gt 0.0 and a_array lt 1e20 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Uranus')
  j = where(r_array gt 1.5 and r_array lt 8 and a_array gt 3.0 and a_array lt 1e20 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Neptune')
  j = where(r_array gt 1.5 and r_array lt 8 and a_array gt 3.0 and a_array lt 1e20 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Warm_Neptune_1AU_Clouds')
  j = where(r_array gt 1.5 and r_array lt 8 and a_array gt 0.0 and a_array lt 2.0 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Warm_Neptune_1AU_NoClouds')
  j = where(r_array gt 1.5 and r_array lt 8 and a_array gt 0.0 and a_array lt 2.0 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif
  
  k = where(files eq 'Warm_Neptune_2AU')
  j = where(r_array gt 1.5 and r_array lt 8 and a_array gt 1.0 and a_array lt 4.0 and EEC ne 1)  
  prob = 100 ;relative weighting of all planets in this phase space (out of 100)
  if k[0] eq -1 then stop,'Error in naming convention'
  if j[0] ne -1 then begin
     j2 = array_indices(r_array,j)
     valid[j2[0,*],j2[1,*],k[0]*100:(k[0]*100+prob-1)] = k[0]     ;record the index of this valid planet prob times
  endif

  nfile_array = intarr(xpix,ypix)
  for i=0,xpix-1 do begin
     for j=0,ypix-1 do begin
        ;calculate total # of valid planet files in this pixel
        nfile_array[i,j] = total(valid[i,j,*] ge 0)
        ;reorder the indeces in each pixel to put -1s at end
        temp = valid[i,j,*]
        temp = temp[reverse(sort(temp))]
        valid[i,j,*] = temp
     endfor
  endfor
  

  for i=0, nstars-1 do begin
     s = stars[i]
     p = s.planet

     ;It's faster to index the planets in mass-semi-major axis space instead
     ;of radius, because it has uniform ln spacing, so no logic is needed.
     tempa = p.a / sqrt(s.Lstar) ;flux-adjusted semi-major axis
     
     planets_lna = alog(tempa)
     planets_lnM = alog(p.M) 
     xindex = (planets_lna - minlna)/dlna
     jx = where(finite(xindex) ne 1) ;these are planets that don't exist. Remember them...
     if jx[0] ne -1 then xindex[jx] = 0 ;temporarily make their indicies friendly
     yindex = (planets_lnM - minlnM)/dlnM
     jy = where(finite(yindex) ne 1) ;these are planets that don't exist. Remember them...
     if jy[0] ne -1 then yindex[jy] = 0 ;temporarily make their indices friendly
     if min(xindex) lt 0 or max(xindex) ge xpix then stop,'Error: planet indices outside of x array boundaries somehow...'
     if min(yindex) lt 0 or max(yindex) ge ypix then stop,'Error: planet indices outside of y array boundaries somehow...'
     ;Now that we have the indices, we can assign albedo files
     rand_index = fix(randomu(seed,n_elements(p)) * nfile_array[xindex,yindex]) ;random index of 3rd dimension of valid[*,*,*]
     random_file_indices = valid[xindex,yindex,rand_index]
     stars[i].planet[*].albedo_file = directory + files[random_file_indices] + '.txt'

     ;Erase albedo files assigned to planets that didn't exist
     if jx[0] ne -1 then stars[i].planet[jx].albedo_file = ''
     if jy[0] ne -1 then stars[i].planet[jy].albedo_file = ''
     
  endfor
 
end
